'use client';

import React, { useState, useEffect } from 'react';
import { useRouter, useParams } from 'next/navigation';
import {
  Shield,
  ArrowLeft,
  Save,
  AlertCircle,
  User,
  Building2,
  CheckCircle
} from 'lucide-react';
import Link from 'next/link';
import type { AgencyUser, AgencyPermission, Department } from '@/types';
import { agencyUserApi, departmentApi } from '@/services/api';
import PermissionSelector from '@/components/permissions/PermissionSelector';

export default function UserPermissionsPage() {
  const router = useRouter();
  const params = useParams();
  const userId = params?.id as string;

  const [user, setUser] = useState<AgencyUser | null>(null);
  const [department, setDepartment] = useState<Department | null>(null);
  const [permissions, setPermissions] = useState<AgencyPermission[]>([]);
  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const [saveSuccess, setSaveSuccess] = useState(false);

  useEffect(() => {
    if (userId) {
      fetchUser();
    }
  }, [userId]);

  const fetchUser = async () => {
    try {
      setLoading(true);

      // Fetch user from API
      const data = await agencyUserApi.getAgencyUser(userId);

      const userWithDates: AgencyUser = {
        id: data.id,
        firstName: data.first_name,
        lastName: data.last_name,
        email: data.email,
        phone: data.phone,
        avatar: data.avatar,
        departmentId: data.department_id,
        department: {
          id: data.department_id,
          name: data.department_name || '',
          color: data.department_color || 'bg-gray-500',
          description: '',
          isActive: true,
          userCount: 0,
          createdAt: new Date(),
          updatedAt: new Date()
        },
        role: data.role,
        permissions: data.permissions ? (typeof data.permissions === 'string' ? JSON.parse(data.permissions) : data.permissions) : [],
        isActive: Boolean(data.is_active),
        isOnline: Boolean(data.is_online),
        lastLoginAt: data.last_login_at ? new Date(data.last_login_at) : undefined,
        managerId: data.manager_id,
        totalTickets: data.total_tickets || 0,
        totalRevenue: data.total_revenue || 0,
        createdAt: new Date(data.created_at),
        updatedAt: new Date(data.updated_at)
      };

      setUser(userWithDates);
      setPermissions(userWithDates.permissions);

      // Fetch department details
      if (data.department_id) {
        const deptData = await departmentApi.getDepartment(data.department_id);
        setDepartment(deptData);
      }
    } catch (error) {
      console.error('Error fetching user:', error);
      alert('Kullanıcı bilgileri yüklenirken bir hata oluştu');
    } finally {
      setLoading(false);
    }
  };

  const handleSavePermissions = async () => {
    if (!user) return;

    try {
      setSaving(true);
      setSaveSuccess(false);

      // Update user permissions via API
      await agencyUserApi.updateAgencyUser(userId, {
        permissions: permissions
      });

      setSaveSuccess(true);
      setTimeout(() => setSaveSuccess(false), 3000);

      // Refresh user data
      await fetchUser();
    } catch (error) {
      console.error('Error updating permissions:', error);
      alert('Yetkiler güncellenirken bir hata oluştu');
    } finally {
      setSaving(false);
    }
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  if (!user) {
    return (
      <div className="p-6">
        <div className="text-center">
          <h2 className="text-2xl font-bold text-gray-900">Kullanıcı bulunamadı</h2>
          <Link href="/users" className="text-blue-600 hover:underline mt-4 inline-block">
            Kullanıcılara geri dön
          </Link>
        </div>
      </div>
    );
  }

  return (
    <div className="p-6">
      {/* Header */}
      <div className="mb-6">
        <Link
          href={`/users/${userId}`}
          className="inline-flex items-center text-sm text-gray-600 hover:text-gray-900 mb-4"
        >
          <ArrowLeft className="h-4 w-4 mr-1" />
          Kullanıcı detayına geri dön
        </Link>

        <div className="flex items-center justify-between">
          <div>
            <div className="flex items-center">
              <Shield className="h-8 w-8 text-blue-600 mr-3" />
              <div>
                <h1 className="text-3xl font-bold text-gray-900">Kullanıcı Yetkileri</h1>
                <p className="text-gray-500 mt-1">
                  {user.firstName} {user.lastName} için yetki düzenlemesi
                </p>
              </div>
            </div>
          </div>

          <button
            onClick={handleSavePermissions}
            disabled={saving}
            className={`flex items-center px-6 py-3 rounded-lg font-medium transition-colors ${
              saving
                ? 'bg-gray-400 cursor-not-allowed'
                : saveSuccess
                ? 'bg-green-600 hover:bg-green-700'
                : 'bg-blue-600 hover:bg-blue-700'
            } text-white`}
          >
            {saving ? (
              <>
                <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-white mr-2"></div>
                Kaydediliyor...
              </>
            ) : saveSuccess ? (
              <>
                <CheckCircle className="h-5 w-5 mr-2" />
                Kaydedildi
              </>
            ) : (
              <>
                <Save className="h-5 w-5 mr-2" />
                Değişiklikleri Kaydet
              </>
            )}
          </button>
        </div>
      </div>

      {/* User Info Card */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 mb-6">
        <h2 className="text-lg font-semibold text-gray-900 mb-4">Kullanıcı Bilgileri</h2>
        <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
          <div className="flex items-start">
            <User className="h-5 w-5 text-gray-400 mr-3 mt-0.5" />
            <div>
              <p className="text-sm font-medium text-gray-500">Kullanıcı</p>
              <p className="text-base font-semibold text-gray-900 mt-1">
                {user.firstName} {user.lastName}
              </p>
              <p className="text-sm text-gray-600">{user.email}</p>
            </div>
          </div>

          <div className="flex items-start">
            <Building2 className="h-5 w-5 text-gray-400 mr-3 mt-0.5" />
            <div>
              <p className="text-sm font-medium text-gray-500">Departman</p>
              <p className="text-base font-semibold text-gray-900 mt-1">
                {user.department.name}
              </p>
              {department && department.permissions && (
                <p className="text-sm text-gray-600">
                  Departman: {Array.isArray(department.permissions) ? department.permissions.length : 0} yetki
                </p>
              )}
            </div>
          </div>

          <div className="flex items-start">
            <Shield className="h-5 w-5 text-gray-400 mr-3 mt-0.5" />
            <div>
              <p className="text-sm font-medium text-gray-500">Mevcut Yetkiler</p>
              <p className="text-base font-semibold text-gray-900 mt-1">
                {permissions.length} yetki
              </p>
              <p className="text-sm text-gray-600">
                {user.isActive ? 'Aktif kullanıcı' : 'Pasif kullanıcı'}
              </p>
            </div>
          </div>
        </div>
      </div>

      {/* Info Alert */}
      {department && department.permissions && (
        <div className="bg-blue-50 border border-blue-200 rounded-lg p-4 mb-6">
          <div className="flex items-start">
            <AlertCircle className="h-5 w-5 text-blue-600 mr-3 mt-0.5" />
            <div>
              <h3 className="text-sm font-medium text-blue-900">Departman Yetkileri</h3>
              <p className="text-sm text-blue-700 mt-1">
                Bu kullanıcı <strong>{user.department.name}</strong> departmanına bağlıdır.
                Departman yetkileri değiştirildiğinde, bu kullanıcının yetkileri de otomatik olarak güncellenir.
              </p>
            </div>
          </div>
        </div>
      )}

      {/* Permissions Selector */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
        <h2 className="text-lg font-semibold text-gray-900 mb-4">Yetki Düzenleme</h2>
        <PermissionSelector
          selectedPermissions={permissions}
          onChange={setPermissions}
          departmentName={user.department.name}
        />
      </div>

      {/* Save Button (Bottom) */}
      <div className="mt-6 flex justify-end">
        <button
          onClick={handleSavePermissions}
          disabled={saving}
          className={`flex items-center px-6 py-3 rounded-lg font-medium transition-colors ${
            saving
              ? 'bg-gray-400 cursor-not-allowed'
              : saveSuccess
              ? 'bg-green-600 hover:bg-green-700'
              : 'bg-blue-600 hover:bg-blue-700'
          } text-white`}
        >
          {saving ? (
            <>
              <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-white mr-2"></div>
              Kaydediliyor...
            </>
          ) : saveSuccess ? (
            <>
              <CheckCircle className="h-5 w-5 mr-2" />
              Kaydedildi
            </>
          ) : (
            <>
              <Save className="h-5 w-5 mr-2" />
              Değişiklikleri Kaydet
            </>
          )}
        </button>
      </div>
    </div>
  );
}
